function [x,y,errrel] = fd_solucion_directa(n)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                        %
% SOLUCION DIRECTA DE UN PROBLEMA DE DIFERENCIAS FINITAS                 %
%                                                                        %
%  [x,y,errrel] = fd_solucion_directa(n)                                 %
%                                                                        %
%  ENTRADA                                                               %
%                                                                        %
%          n    numero de puntos de la malla                             %
%                                                                        %
%  SALIDA                                                                %
%                                                                        %
%         x     malla en X                                               %
%         y     solucion en los puntos de la malla                       %
%    errrel     error relativo (comparado con la solucion analitica)     %
%                                                                        %
% Consideramos la ecuacion y''-5y'+10y=10x con cc y(0)=0, y(1)=100,      %
% aproximada por diferencias finitas de segundo orden. Resolvemos el     %
% sistema de ecuaciones resultante RESOLVIENDOLO DIRECTAMENTE.           %
%                                                                        %
% Se compara la solucion con la solucion analitica. En el caso de que    %
% se disponga de la "Symbolic Math Toolbox" puede hallarse automatica-   %
% mente tal solucion analitica.                                          %
%                                                                        %
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% limpieza del espacio de trabajo
figure(10);clf;

%
% PARAMETROS DEL PROBLEMA ORIGINAL
%
a = 0.0;       % limite inferior del dominio
b = 1.0;       % limites superior del dominio
               % CONDICIONES DE CONTORNO
ya = 0.0;      % valor de la funcion en el limite inferior del dominio
yb = 1.0e2;    % valor de la funcion en el limite superior del dominio

%
% PARAMETROS DEL PROBLEMA DISCRETIZADO
%
%n = 101;           % numero de puntos de la malla

% malla equiespaciada en variable indendiente
x = linspace( a, b, n); 
dx = x(2)-x(1);

% coeficientes de la ODE discretizada
c1 = 1.0 / (2.0 - 10.0*dx^2);
c2 = c1 * ( 1.0 - 2.5*dx);
c3 = c1 * ( 1.0 + 2.5*dx);
c4 = -c1*10.0*dx^2;

% matriz del problema lineal (DIMENSION N-2)
%
% -c3*y(ii-1) + y( ii ) - c2*y(ii+1) = c4*x( ii );
%

dd = ones(1, n-2);
ds = ones(1, n-3)*(-c2);
di = ones(1, n-3)*(-c3);
b  = c4*x( 2:n-1 ); 
b(1) = b(1) + c3*ya;
b(end) = b(end) + c2*yb;
b = b';

Mfd = diag(dd,0) + diag(ds,1) + diag(di,-1);
%yy = linsolve(Mfd,b);
yy = Mfd\b;
y = [ya; yy; yb];
y = y';

% comparamos con la solucion analitica
ff = matlabFunction( dsolve( 'D2y - 5*Dy + 10*y = 10*x','y(0)=0', ...
        'y(1)=100','x') );
errrel = max( abs( (y-ff(x))/y ));
fprintf('\n error relativo del resultado %e\n', errrel);

% dibujamos la solucion numerica y la analitica
figure(10);
plot(x,y,'ro',x,ff(x),'-b');
xlabel('x'); ylabel('y');
legend('Solucion directa','Solucion analitica','Location','North');
